package coins.backend.util;

import java.util.*;
import java.io.*;
import coins.backend.*;

/**
 * Bi-directional list (head part)
 */
public class BiList extends BiLink {

  /** Create empty list. */
  public BiList() {
    next = this; // First node
    prev = this; // Last node
  }

  public Object elem() {
    throw new UnsupportedOperationException();
  }

  public BiLink unlink() {
    throw new UnsupportedOperationException();
  }

  public BiLink next() {
    throw new UnsupportedOperationException("attempt to next past end");
  }

  public BiLink prev() {
    throw new UnsupportedOperationException("attempt to prev past end");
  }

  /** Return first link. */
  public BiLink first() { return next; }

  /** Return last link. */
  public BiLink last() { return prev; }

  /** Return true if this link is either end. */
  public boolean atEnd() { return true; }

  /** Return true if this list is empty. */
  public boolean isEmpty() { return next == this; }

  /** Append an element to the end of the list. */
  public BiLink add(Object obj) {
    return this.addBefore(obj);
  }

  /** Append a link to the end of the list. */
  public BiLink append(BiLink link) {
    return this.insertBefore(link);
  }

  /** Add an element before first element of the list. */
  public BiLink addFirst(Object obj) {
    return this.addAfter(obj);
  }

  /** Prepend a link before first element of the list. */
  public BiLink prepend(BiLink link) {
    return this.insertAfter(link);
  }

  /** Test if the list contains an object obj.  */
  public boolean contains(Object obj) {
    for (BiLink p = next; p != this; p = p.next) {
      if (p.elem == obj)
        return true;
    }
    return false;
  }

  /** Find a link which has an object obj.  */
  public BiLink locate(Object obj) {
    BiLink p;
    for (p = next; p != this; p = p.next) {
      if (p.elem == obj)
        return p;
    }
    return null;
  }

  /** Remove an link which has an object obj. */
  public BiLink remove(Object obj) {
    BiLink p = locate(obj);
    if (p != null)
      p.unlink();
    return p;
  }

  /** Add an object to the list only if not there. */
  public BiList addNew(Object obj) {
    if (!contains(obj))
      add(obj);
    return this;
  }

  /** Clear the list. Make list empty. */
  public void clear() {
    next = prev = this;
  }

  /** Concatenate two lists. Added list will be destroyed.
   * @param aList the list appended
   * @return this object itself */
  public BiList concatenate(BiList aList) {
    prev.next = aList.next;
    aList.next.prev = prev;
    prev = aList.prev;
    prev.next = this;
    // make aList empty (not necessary but safer)
    aList.next = aList.prev = null;
    return this;
  }

  /** Split the list into two parts.
   * @param middle the first element of second half of the list
   * @return second half of the list (first half is <code>this</code> object) */
  public BiList split(BiLink middle) {
    BiList secondHalf = new BiList();
    if (middle != this) {
      secondHalf.next = middle;
      secondHalf.prev = prev;
      prev.next = secondHalf;
      prev = middle.prev;
      middle.prev = secondHalf;
      prev.next = this;
    }
    return secondHalf;
  }

  /** Make a copy of the list and return it.
   * Do not copy the contents; they are shared. */
  public BiList copy() {
    BiList newList = new BiList();
    for (BiLink p = next; p != this; p = p.next)
      newList.add(p.elem);
    return newList;
  }

  /** Return length of the list */
  public int length() {
    int n = 0;
    for (BiLink p = next; p != this; p = p.next) {
      if (p.atEnd()) {
        throw new Error("abnormal list: tail and head do not match");
      }
      n++;
    }
    return n;
  }

  /** Visualize */
  public String toString() {
    StringBuffer buf = new StringBuffer();
    buf.append("(");
    boolean first = true;
    for (BiLink p = next; p != this; p = p.next) {
      if (!first) buf.append(" ");
      buf.append(p.elem.toString());
      first = false;
    }
    buf.append(")");
    return buf.toString();
  }


  /** Iterator for scanning a BiList. */
  class BiListIterator implements Iterator {
    BiLink current;

    BiListIterator(BiList list) { current = list.next; }

    public boolean hasNext() { return !current.atEnd(); }

    public Object next() {
      Object content = current.elem;
      current = current.next;
      return content;
    }

    public void remove() { throw new UnsupportedOperationException(); }
  }

  /** Return iterator for the list. */
  public Iterator iterator() {
    return new BiListIterator(this);
  }
}
