package coins.backend.ana;

import java.io.*;
import java.util.*;
import coins.backend.*;
import coins.backend.cfg.*;
import coins.backend.lir.*;
import coins.backend.sym.*;
import coins.backend.util.*;

/** Live variable analysis.
 *  Set information is represented by bitmap vectors. */
public class LiveVariableAnalysis extends DataFlowAnalysis {

  /** Factory class of LiveVariableAnalysis. */
  private static class Analyzer implements LocalAnalyzer {
    public LocalAnalysis doIt(Function func) {
      return new LiveVariableAnalysis(func);
    }
  }


  /** Factory singleton. */
  public static final Analyzer analyzer = new Analyzer();


  /** Live information for each basic block **/
  private BitSet [] in;
  private BitSet [] out;
  private BitSet [] gen;
  private BitSet [] kill;

  // private BitSet [] liveAt;

  private int maxSyms;
  private Symbol[] symVec;


  /** Construct live variable information */
  public LiveVariableAnalysis(Function f) {
    super(f);
  }


  
  /* Interface of Live Variable Analysis */

  /** Return true if variable x is live at position (blk, stmt). **/
  /*
  public boolean isLiveAt(Symbol regvar, BasicBlk blk, LirNode stmt) {
    return liveAt[stmt.id].get(regvar.id);
  }
  */


  /** Return true if variable regvar is live at entry of blk. **/
  public boolean isLiveAtEntry(Symbol regvar, BasicBlk blk) {
    return in[blk.id].get(regvar.id);
  }

  /** Return true if variable regvar is live at exit of blk. **/
  public boolean isLiveAtExit(Symbol regvar, BasicBlk blk) {
    return out[blk.id].get(regvar.id);
  }

  /** Return the list of live variables at position (blk, stmt). **/
  /*
  public BiList liveAt(BasicBlk blk, LirNode stmt) {
    BiList live = new BiList();
    for (int i = 0; (i = liveAt[stmt.id].nextSetBit(i)) >= 0; i++)
      live.addNew(symVec[i]);
    return live;
  }
  */


  /** Return the list of live variables at exit of basic block blk. **/
  public BiList liveOut(BasicBlk blk) {
    BiList live = new BiList();
    for (int i = 0; (i = out[blk.id].nextSetBit(i)) >= 0; i++)
      live.addNew(symVec[i]);
    return live;
  }


  /** Return the list of live variables at entry of basic block blk. **/
  public BiList liveIn(BasicBlk blk) {
    BiList live = new BiList();
    for (int i = 0; (i = in[blk.id].nextSetBit(i)) >= 0; i++)
      live.addNew(symVec[i]);
    return live;
  }

  

  /** Return BitSet of live variables at exit of basic block blk. **/
  BitSet liveOutBitSet(BasicBlk blk) {
    return (BitSet)out[blk.id].clone();
  }


  /** Return BitSet of live variables at entry of basic block blk. **/
  BitSet liveInBitSet(BasicBlk blk) {
    return (BitSet)in[blk.id].clone();
  }


  /** Maintain live set over a statement backward. **/
  void maintainLiveOverStmt(final BitSet live, LirNode stmt) {
    pickupDefUseReg(stmt, new DefUseHandler() {
        public void defined(LirNode node) {
          doDef(null, live, node);
        }

        public void used(LirNode node) {
          doUse(null, live, node);
        }
      });
  }
  
    


  /* Problem-Oriented Methods called by super class (DataFlowAnalysis). */

  /** Initialize problem-oriented data structure. **/
  void initialize() {
    isForward = false; // live variable anal. is backward-direction problem

    int maxBlks = function.flowGraph.idBound();
    maxSyms = function.localSymtab.idBound();
    symVec = function.localSymtab.sortedSymbols();
    in = new BitSet[maxBlks];
    out = new BitSet[maxBlks];
    kill = new BitSet[maxBlks];
    gen = new BitSet[maxBlks];
    for (int i = 0; i < maxBlks; i++) {
      in[i] = new BitSet(maxSyms);
      out[i] = new BitSet(maxSyms);
    }

    // Compute kill and gen.
    for (BiLink p = function.flowGraph.basicBlkList.first(); !p.atEnd();
         p = p.next()) {
      BasicBlk blk = (BasicBlk)p.elem();
      kill[blk.id] = new BitSet(maxSyms);
      gen[blk.id] = new BitSet(maxSyms);

      // Scan LirNodes list backward
      for (BiLink q = blk.instrList().last(); !q.atEnd(); q = q.prev()) {
        LirNode stmt = (LirNode)q.elem();
        if (stmt.opCode != Op.PHI)
          doStmt(stmt, kill[blk.id], gen[blk.id]);
      }
    }

  }



  /** Maintain kill/gen sets over a statement. **/
  private void doStmt(LirNode stmt, final BitSet kill, final BitSet gen) {

    pickupDefUseReg(stmt, new DefUseHandler() {
        public void defined(LirNode node) {
          doDef(kill, gen, node);
        }

        public void used(LirNode node) {
          doUse(kill, gen, node);
        }
      });
  }



  /** Is register operand? **/
  private static boolean isRegisterOperand(LirNode ins) {
    return (ins.opCode == Op.REG
            || ins.opCode == Op.SUBREG && ins.src(0).opCode == Op.REG);
  }

  /** Pick up def'd/used register variables in the LirNode tree. **/
  static void pickupDefUseReg(LirNode tree, DefUseHandler handler) {
    switch (tree.opCode) {
    case Op.PARALLEL:
      int n = tree.nSrcs();
      for (int i = 0; i < n; i++) {
        switch (tree.src(i).opCode) {
        case Op.SET:
        case Op.CLOBBER:
          if (isRegisterOperand(tree.src(i).src(0)))
            handler.defined(tree.src(i).src(0));
          break;
        }
      }
      for (int i = 0; i < n; i++) {
        switch (tree.src(i).opCode) {
        case Op.SET:
          if (!isRegisterOperand(tree.src(i).src(0)))
            pickupDefUseReg(tree.src(i).src(0), handler);
          pickupDefUseReg(tree.src(i).src(1), handler);
          break;
        case Op.CLOBBER:
          break;
        default:
          {
            int m = tree.src(i).nSrcs();
            for (int j = 0; j < m; j++)
              pickupDefUseReg(tree.src(i).src(j), handler);
          }
          break;
        }
      }
      break;

    case Op.SET:
      if (isRegisterOperand(tree.src(0)))
        handler.defined(tree.src(0));
      else
        pickupDefUseReg(tree.src(0), handler);
      pickupDefUseReg(tree.src(1), handler);
      break;

    case Op.CLOBBER:
      if (isRegisterOperand(tree.src(0)))
        handler.defined(tree.src(0));
      break;

      // case Op.CALL:

    default:
      if (isRegisterOperand(tree))
        handler.used(tree);
      else {
        int m = tree.nSrcs();
        for (int j = 0; j < m; j++)
          pickupDefUseReg(tree.src(j), handler);
      }
      break;
    }
  }



  /** Maintain kill/gen sets when the node is a register variable read. **/
  private void doUse(BitSet kill, BitSet gen, LirNode node) {
    if (node.opCode == Op.SUBREG)
      node = node.src(0);
    if (node.opCode == Op.REG) {
      int refd = ((LirSymRef)node).symbol.id;
      if (kill != null) kill.set(refd, false);
      gen.set(refd);
    }
  }


  /** Maintain kill/gen sets when the node is a register variable written. **/
  private void doDef(BitSet kill, BitSet gen, LirNode node) {
    if (node.opCode == Op.SUBREG)
      node = node.src(0);
    if (node.opCode == Op.REG) {
      int defd = ((LirSymRef)node).symbol.id;
      if (kill != null) kill.set(defd);
      gen.set(defd, false);
    }
  }



  /** Supply confluence operator (=join at here) for block blk. **/
  void confluence(BasicBlk blk) {
    out[blk.id].clear();
    for (BiLink s = blk.succList().first(); !s.atEnd(); s = s.next()) {
      BasicBlk succ = (BasicBlk)s.elem();
      out[blk.id].or(in[succ.id]);

      /** Special treatment of PHI functions. **/
      /** x0 = phi(x1, x2*, x3); --> kill x0, gen x2 only if x0 is live **/

      for (BiLink q = succ.instrList().first(); !q.atEnd(); q = q.next()) {
        LirNode stmt = (LirNode)q.elem();
        if (stmt.opCode != Op.PHI)
          break;

        // Mark RHS 'used' only if LHS of PHI is live.
        if (out[blk.id].get(((LirSymRef)stmt.src(0)).symbol.id)) {
          // Scan RHS of PHI which belongs to this block.
          int n = stmt.nSrcs();
          for (int i = 1; i < n; i++) {
            if (stmt.src(i).src(1).opCode == Op.LABEL
                && ((LirLabelRef)stmt.src(i).src(1)).label == blk.label()
                && stmt.src(i).src(0).opCode == Op.REG)
              out[blk.id].set(((LirSymRef)stmt.src(i).src(0)).symbol.id);
          }
        }

        // Kill LHS of PHI.
        out[blk.id].set(((LirSymRef)stmt.src(0)).symbol.id, false);
      }
    }
  }

  /** Supply transfer function for block blk. */
  boolean transfer(BasicBlk blk) {
    BitSet newin = (BitSet)out[blk.id].clone();
    newin.andNot(kill[blk.id]);
    newin.or(gen[blk.id]);
    boolean changed = !newin.equals(in[blk.id]);
    in[blk.id] = newin;
    return changed;
  }


  /** Finalization. Set up local-live information. **/
  void windUp() {
    /*
    liveAt = new BitSet[function.newLir.idBound()];

    for (BiLink p = function.flowGraph.basicBlkList.first(); !p.atEnd();
         p = p.next()) {
      BasicBlk blk = (BasicBlk)p.elem();
      BitSet live = (BitSet)out[blk.id].clone();

      for (BiLink q = blk.instrList().last(); !q.atEnd(); q = q.prev()) {
        LirNode stmt = (LirNode)q.elem();
        liveAt[stmt.id] = (BitSet)live.clone();
        if (stmt.opCode != Op.PHI)
          doStmt(stmt, null, live);
      }
    }
    */
  }

  /** Print Live variables */
  private void printLive(String head, BitSet live, PrintWriter output) {
    output.print(head);
    for (int i = 0; (i = live.nextSetBit(i)) >= 0; i++)
      output.print(" " + symVec[i].name);
    output.println();
  }

  /** Print internal state for debugging (OBSOLETED). **/
  public void printIt(PrintWriter out) {  }

  /** Debug print entries required by interface. **/

  public void printBeforeFunction(PrintWriter output) {}

  public void printBeforeBlock(BasicBlk blk, PrintWriter output) {
    printLive("   kill:", kill[blk.id], output);
    printLive("    gen:", gen[blk.id], output);
    printLive("    in:", in[blk.id], output);
  }

  public void printAfterBlock(BasicBlk blk, PrintWriter output) {
    printLive("   out:", out[blk.id], output);
  }

  public void printBeforeStmt(LirNode stmt, PrintWriter output) {}

  public void printAfterStmt(LirNode stmt, PrintWriter output) {
    //    printLive("  live:", liveAt[stmt.id], output);
  }

  public void printAfterFunction(PrintWriter output) {}

}
